// Migration script to set periodDate for existing TaskItemCompletion records
// For non-recurring tasks, set periodDate to task's createdAt date
// For recurring tasks, we'll need to calculate the period date

const { PrismaClient } = require("@prisma/client");
const prisma = new PrismaClient();

async function migrate() {
  console.log("Starting migration...");

  // Get all task item completions without periodDate
  const completions = await prisma.taskItemCompletion.findMany({
    where: {
      periodDate: null,
    },
    include: {
      task: true,
    },
  });

  console.log(`Found ${completions.length} completions to migrate`);

  for (const completion of completions) {
    const task = completion.task;

    if (task.isRecurring) {
      // For recurring tasks, calculate the period date based on recurrence pattern
      // Use the checkedAt date to determine which period it belongs to
      const checkedDate = new Date(completion.checkedAt);
      let periodDate = new Date(checkedDate);

      if (task.recurrencePattern === "daily") {
        periodDate.setHours(0, 0, 0, 0);
      } else if (task.recurrencePattern === "weekly") {
        // Start of week (Monday)
        const day = checkedDate.getDay();
        const diff = checkedDate.getDate() - day + (day === 0 ? -6 : 1);
        periodDate = new Date(checkedDate.setDate(diff));
        periodDate.setHours(0, 0, 0, 0);
      } else if (task.recurrencePattern === "monthly") {
        // Start of month
        periodDate = new Date(
          checkedDate.getFullYear(),
          checkedDate.getMonth(),
          1
        );
        periodDate.setHours(0, 0, 0, 0);
      }

      await prisma.taskItemCompletion.update({
        where: { id: completion.id },
        data: { periodDate },
      });
    } else {
      // For non-recurring tasks, use task's createdAt date (normalized)
      const taskCreatedDate = new Date(task.createdAt);
      taskCreatedDate.setHours(0, 0, 0, 0);

      await prisma.taskItemCompletion.update({
        where: { id: completion.id },
        data: { periodDate: taskCreatedDate },
      });
    }
  }

  console.log("Migration completed!");
}

migrate()
  .catch((e) => {
    console.error("Migration failed:", e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
